// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import MSSOrchestration

class LocalAuthenticationViewController: UIViewController {
    @IBOutlet weak var userIdLabel: UILabel!
    @IBOutlet weak var userId: UILabel!
    @IBOutlet weak var challengeLabel: UILabel!
    @IBOutlet weak var challenge: UITextField!
    @IBOutlet weak var challengeError: UILabel!
    @IBOutlet weak var protectionTitle: UILabel!
    @IBOutlet weak var noPasswordButton: UIButton!
    @IBOutlet weak var passwordButton: UIButton!
    @IBOutlet weak var biometricButton: UIButton!
    @IBOutlet weak var generateOTPButton: UIButton!
    @IBOutlet weak var scroll: UIScrollView!
    
    private let orchestrationDelegate = OrchestrationSampleDelegate()
    private let userAuthenticationDelegate = UserAuthenticationViewController()
    
    private var protectionSelected: UIButton?
    private var responder: UIView?
    private var progressDialog: UIView?
    private var orchestrator: Orchestrator?
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupView()
        setupOrchestration()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(keyboardWillShow(_:)),
                                               name: UIResponder.keyboardWillShowNotification,
                                               object: nil)
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(keyboardWillHide(notification:)),
                                               name: UIResponder.keyboardWillHideNotification,
                                               object: nil)
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        dismissKeyboard()
        NotificationCenter.default.removeObserver(self)
    }
    
    // MARK: Setup
    private func setupView() {
        title = "title_activity_local_authentication".localized
        navigationItem.hidesBackButton = false
        
        userIdLabel.text = "prompt_user_id".localized
        challengeLabel.text = "prompt_challenge".localized
        
        protectionTitle.text = "protection_title".localized
        noPasswordButton.setTitle("protection_type_no_password".localized, for: UIControl.State.normal)
        passwordButton.setTitle("protection_type_password".localized, for: UIControl.State.normal)
        biometricButton.setTitle("protection_type_biometric".localized, for: UIControl.State.normal)
        
        generateOTPButton.setTitle("btn_generate_otp".localized, for: UIControl.State.normal)
        
        navigationController?.navigationItem.accessibilityLabel = "backButton"
        
        protectionSelected = noPasswordButton
        protectionSelected?.isSelected = true
        
        // Scroll and form configuration
        let gesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        scroll.addGestureRecognizer(gesture)
        
        generateOTPButton.layer.cornerRadius = 3
        
        userId.text = SharedPreferenceStorage.getActivatedUser()
        
        challenge.delegate = self
    }
    
    private func setupOrchestration() {
        orchestrationDelegate.progressDialog = progressDialog
        orchestrationDelegate.viewController = self
        orchestrator = OrchestratorUtils.getOrchestrator(delegate: orchestrationDelegate)
        
        // Used for custom password instead of default one
        guard let orchestrator = orchestrator else {
            assertionFailure("orchestrator in LocalAuthenticationViewController is nil")
            return
        }
        
        userAuthenticationDelegate.useExternalPassword(orchestrator: orchestrator, viewController: self)
    }
    
    // MARK: IBActions
    @IBAction func onProtectionButtonClick(_ sender: Any) {
        protectionSelected?.isSelected = false
        protectionSelected = sender as? UIButton
        protectionSelected?.isSelected = true
    }
    
    @IBAction func onGenerateOTPButtonClick(_ sender: Any) {
        challengeError.text = ""
        let challenge = self.challenge.text
        
        if let challenge = challenge, !challenge.isEmpty {
            let regexp = "[0-9a-fA-F]{6}"
            let regexPredicate = NSPredicate(format: "SELF MATCHES %@", regexp)
            if !regexPredicate.evaluate(with: challenge) {
                challengeError.text = "error_local_auth_challenge_invalid".localized
                return
            }
        }
        guard let identifier = userId.text else {
            assertionFailure("userId text is nil")
            return
        }
        let orchestrationUser = OrchestrationUser(identifier: identifier, domain: nil)
        let cryptoAppIndex: CryptoAppIndex
        if challenge == nil || challenge?.isEmpty == true {
            // Second crypto app is used when no challenge is needed
            cryptoAppIndex = .app2
        } else {
            // Third crypto app is used when challenge is needed
            cryptoAppIndex = .app3
        }
        let protectionType = getSelectedProtectionType()
        let parameters = LocalAuthenticationParameters(user: orchestrationUser,
                                                       cryptoAppIndex: cryptoAppIndex,
                                                       challenge: challenge ?? "",
                                                       protectionType: protectionType,
                                                       delegate: self)
        
        progressDialog = UIUtils.displayProgress(controller: self, message: "dialog_progress_local_auth".localized)
        dismissKeyboard()
        orchestrationDelegate.progressDialog = progressDialog
        orchestrator?.startLocalAuthentication(with: parameters)
    }
    
    // MARK: Protection selection
    private func getSelectedProtectionType() -> ProtectionType {
        if passwordButton.isSelected { return .password }
        if biometricButton.isSelected { return .biometric }
        return .noPassword
    }
}

extension LocalAuthenticationViewController: UITextFieldDelegate {
    func textFieldDidBeginEditing(_ textField: UITextField) {
        responder = textField
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        // remove focus from current textfield
        textField.resignFirstResponder()
        return true
    }
}

// MARK: Keyboard event
extension LocalAuthenticationViewController {
    @objc private func keyboardWillShow(_ notification: Notification) {
        if let keyboardFrame = notification.userInfo?[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue {
            let keyboardRectangle = keyboardFrame.cgRectValue
            let keyboardSize = keyboardRectangle.size
            let contentInsets = UIEdgeInsets(top: 0.0, left: 0.0, bottom: keyboardSize.height, right: 0.0)
            scroll.scrollIndicatorInsets = contentInsets
            scroll.contentInset = contentInsets
            
            var visibleFrame = scroll.frame
            visibleFrame.size.height -= keyboardSize.height
            if let responder = responder {
                let cellRect = scroll.convert(responder.frame, from: responder.superview)
                if !visibleFrame.contains(cellRect) {
                    scroll.scrollRectToVisible(cellRect, animated: true)
                }
            }
        }
    }
    
    @objc private func keyboardWillHide(notification: NSNotification) {
        scroll.scrollsToTop = true
        scroll.contentInset = UIEdgeInsets.zero
        scroll.scrollIndicatorInsets = UIEdgeInsets.zero
    }
    
    @objc private func dismissKeyboard() {
        responder?.resignFirstResponder()
        responder = nil
    }
}

extension LocalAuthenticationViewController: LocalAuthenticationDelegate {
    func orchestrator(_ orchestrator: Orchestrator, didFinishLocalAuthenticationSuccessfullyWith otp: String, hostCode: String?) {
        UIUtils.hideProgress(progressDialog)
        if hostCode?.isEmpty == false {
            UIUtils.displayAlert(controller: self,
                                 title: "dialog_title_local_auth".localized,
                                 message: String(format: "dialog_content_local_auth_success_success_host".localized,
                                                 otp,
                                                 hostCode ?? ""))
        } else {
            UIUtils.displayAlert(
                controller: self,
                title: "dialog_title_local_auth".localized,
                message: String(format: "dialog_content_local_auth_success_success".localized,
                                otp))
        }
    }
    
    func orchestratorDidAbortLocalAuthentication(_ orchestrator: Orchestrator) {
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(controller: self,
                             title: "dialog_title_local_auth".localized,
                             message: "dialog_content_local_auth_abortion".localized)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveLocalAuthenticationError error: OrchestrationError) {
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(for: error, on: self)
    }
}
